program IsDelphiNet;

{$APPTYPE CONSOLE}

uses
  System.IO, System.Reflection;

type
  TAssemblyKind = (akNotFound, akLoadError, akNotClr, akNonDelphi, akDelphi);

function DetectDelphiAssembly(const FileName: string): TAssemblyKind;
var
  A: Assembly;

  function DelphiAssemblyDetected: Boolean;
  var
    I: Integer;
    T: array of System.Type;
    R: array of AssemblyName;
  begin
    Result := False;
    T := A.GetTypes;
    for I := Low(T) to High(T) do
      if T[I].Namespace.StartsWith('Borland.Delphi.System') then
      begin
        Result := True;
        Break;
      end;
    if not Result then
    begin
      R := A.GetReferencedAssemblies;
      for I := Low(R) to High(R) do
        if R[I].Name.StartsWith('Borland.Delphi') then
        begin
          Result := True;
          Break;
        end;
    end;
  end;

begin
  try
    A := Assembly.LoadFrom(FileName);
    if DelphiAssemblyDetected then
      Result := akDelphi
    else
      Result := akNonDelphi;
  except
    on E: BadImageFormatException do
      Result := akNotClr;
    on E: IOException do
      Result := akNotFound;
    on E: ReflectionTypeLoadException do
      Result := akLoadError;
  end;
end;

procedure Run;
var
  FileName: string;
begin
  FileName := ParamStr(1);
  if FileName = '' then
    WriteLn('Usage: isdelphinet <assembly file name>')
  else
    case DetectDelphiAssembly(FileName) of
      akNotFound:
        WriteLn('Assembly not found');
      akLoadError:
        Writeln('Can not load assembly, some referenced assemblies are probably missing');
      akNotClr:
        WriteLn('Specified file is not CLR');
      akNonDelphi:
        WriteLn('Assembly does not seem to be created by Delphi for .NET');
      akDelphi:
        WriteLn('Assembly is written in Delphi for .NET');
    end;
end;

begin
  Run;
end.
